<?php
    
    namespace common\models;
    
    use Yii;
    
    /**
     * This is the model class for table "road_service".
     * @property integer       $road_service_id
     * @property integer       $road_id
     * @property integer       $region_id
     * @property double        $begin
     * @property double        $end
     * @property integer       $road_direction_id
     * @property integer       $organization_id
     * @property integer       $year_begin
     * @property Organization  $organization
     * @property Region        $region
     * @property Road          $road
     * @property RoadDirection $roadDirection
     */
    class RoadService extends \yii\db\ActiveRecord
    {
        
        /**
         * @inheritdoc
         */
        public static function tableName()
        {
            return 'road_service';
        }
        
        /**
         * @inheritdoc
         */
        public function rules()
        {
            return [
                [
                    [
                        'road_id',
                        'region_id',
                        'begin',
                        'end',
                        'organization_id',
                        'year_begin',
                    ],
                    'required',
                ],
                [
                    [
                        'road_id',
                        'region_id',
                        'road_direction_id',
                        'organization_id',
                    ],
                    'integer',
                ],
                [
                    [
                        'begin',
                        'end',
                    ],
                    'number',
                ],
                [
                    [ 'year_begin' ],
                    'integer',
                    'min' => 1991,
                    'max' => date('Y'),
                ],
                [
                    [ 'organization_id' ],
                    'exist',
                    'skipOnError'     => true,
                    'targetClass'     => Organization::className(),
                    'targetAttribute' => [ 'organization_id' => 'organization_id' ],
                ],
                [
                    [ 'region_id' ],
                    'exist',
                    'skipOnError'     => true,
                    'targetClass'     => Region::className(),
                    'targetAttribute' => [ 'region_id' => 'region_id' ],
                ],
                [
                    [ 'road_id' ],
                    'exist',
                    'skipOnError'     => true,
                    'targetClass'     => Road::className(),
                    'targetAttribute' => [ 'road_id' => 'road_id' ],
                ],
                [
                    [ 'road_direction_id' ],
                    'exist',
                    'skipOnError'     => true,
                    'targetClass'     => RoadDirection::className(),
                    'targetAttribute' => [ 'road_direction_id' => 'road_direction_id' ],
                ],
            ];
        }
        
        /**
         * @inheritdoc
         */
        public function attributeLabels()
        {
            return [
                'road_service_id'   => 'Індекс',
                'road_id'           => 'Автомобільна дорога',
                'region_id'         => 'Область',
                'begin'             => 'Місцезнаходження, км+ початок',
                'end'               => 'Місцезнаходження, км+ кінець',
                'road_direction_id' => 'Напрямок смуги руху',
                'organization_id'   => 'Назва організації чи підприємства, що обслуговує ділянку дороги',
                'year_begin'        => 'Рік початку обслуговування',
            ];
        }
        
        /**
         * @return \yii\db\ActiveQuery
         */
        public function getOrganization()
        {
            return $this->hasOne(Organization::className(), [ 'organization_id' => 'organization_id' ])
                        ->inverseOf('roadServices');
        }
        
        /**
         * @return \yii\db\ActiveQuery
         */
        public function getRegion()
        {
            return $this->hasOne(Region::className(), [ 'region_id' => 'region_id' ])
                        ->inverseOf('roadServices');
        }
        
        /**
         * @return \yii\db\ActiveQuery
         */
        public function getRoad()
        {
            return $this->hasOne(Road::className(), [ 'road_id' => 'road_id' ])
                        ->inverseOf('roadServices');
        }
        
        /**
         * @return \yii\db\ActiveQuery
         */
        public function getRoadDirection()
        {
            return $this->hasOne(RoadDirection::className(), [ 'road_direction_id' => 'road_direction_id' ])
                        ->inverseOf('roadServices');
        }
        
        public function getTitle()
        {
            $title = '';
            $title .= $this->road->name;
            $title .= ' км ' . $this->getBeginString() . ' - км ' . $this->getEndString();
            return $title;
        }
        
        public function getBeginString()
        {
            return floor($this->begin) . '+' . ( str_pad(( ( $this->begin - floor($this->begin) ) * 1000 ), 3, '0', STR_PAD_LEFT) );
        }
        
        public function getEndString()
        {
            return floor($this->end) . '+' . ( str_pad(( ( $this->end - floor($this->end) ) * 1000 ), 3, '0', STR_PAD_LEFT) );
        }
    }
