<?php
    
    namespace frontend\controllers;
    
    use artbox\catalog\models\Brand;
    use artbox\catalog\models\Category;
    use artbox\catalog\models\Product;
    use artbox\core\models\DummyAlias;
    use artbox\core\models\Feedback;
    use artbox\order\models\Customer;
    use artbox\order\models\LoginForm;
    use artbox\order\models\PasswordResetRequestForm;
    use artbox\order\models\ResetPasswordForm;
    use artbox\order\models\SignupForm;
    use artbox\weblog\models\Article;
    use common\models\Settings;
    use frontend\models\ShareBasket;
    use nodge\eauth\ErrorException;
    use nodge\eauth\openid\ControllerBehavior;
    use Yii;
    use yii\base\InvalidParamException;
    use yii\bootstrap\Html;
    use yii\helpers\Json;
    use yii\web\BadRequestHttpException;
    use yii\web\Controller;
    use yii\filters\VerbFilter;
    use yii\web\Response;
    
    /**
     * Site controller
     */
    class SiteController extends Controller
    {
        /**
         * @inheritdoc
         */
        public function actions()
        {
            return [
                'error' => [
                    'class' => 'yii\web\ErrorAction',
                ],
            ];
        }
        
        /**
         * @inheritdoc
         */
        public function behaviors()
        {
            return [
                'verbs' => [
                    'class'   => VerbFilter::className(),
                    'actions' => [
                        'feedback'     => [ 'post' ],
                        'share-basket' => [ 'post' ],
                    ],
                ],
                'eauth' => [
                    'class' => ControllerBehavior::className(),
                    'only'  => [ 'login' ],
                ],
            ];
        }
        
        /**
         * Displays homepage.
         *
         * @return mixed
         */
        public function actionIndex()
        {
            /**
             * @var \artbox\core\components\SeoComponent $seo
             */
            $seo = \Yii::$app->get('seo');
            $seo->setAlias(
                new DummyAlias(
                    [
                        'title' => \Yii::t('app', 'Главная страница'),
                    ]
                )
            );
            $categories = Category::find()
                                  ->with('categories.lang', 'lang')
                                  ->where([ 'level' => 0 ])
                                  ->all();
            $topItems = Product::find()
                               ->with('lang', 'image', 'variants.image')
                               ->is('mask', 1)
                               ->limit(20)
                               ->all();
            $newItems = Product::find()
                               ->with('lang', 'image', 'variants.image')
                               ->is('mask', 2)
                               ->limit(20)
                               ->all();
            $saleItems = Product::find()
                                ->with('lang', 'image', 'variants.image')
                                ->is('mask', 4)
                                ->limit(20)
                                ->all();
            $historyItems = [];
            /**
             * @var \artbox\catalog\components\History $history
             */
            if ($history = \Yii::$app->get('history', false)) {
                if (!empty($history->get())) {
                    $historyItems = Product::find()
                                           ->with('lang', 'image')
                                           ->innerJoinWith(
                                               [
                                                   'variants' => function ($query) {
                                                       /**
                                                        * @var \yii\db\ActiveQuery $query
                                                        */
                                                       $query->with('image');
                                                   },
                                               ]
                                           )
                                           ->where(
                                               [
                                                   'variant.id' => $history->get(),
                                               ]
                                           )
                                           ->limit(20)
                                           ->all();
                }
            }
            $productCount = Product::find()
                                   ->count();
            $brandCount = Brand::find()
                               ->count();
            $brands = Brand::find()
                           ->where([ 'status' => true ])
                           ->andWhere(
                               [
                                   'not',
                                   [ 'image_id' => null ],
                               ]
                           )
                           ->orderBy([ 'sort' => SORT_ASC ])
                           ->limit(6)
                           ->with('image')
                           ->all();
            $articles = Article::find()
                               ->orderBy('sort')
                               ->with(
                                   [
                                       'lang',
                                       'image',
                                       'category.lang',
                                   ]
                               )
                               ->limit(4)
                               ->all();
            return $this->render(
                'index',
                [
                    'categories'   => $categories,
                    'topItems'     => $topItems,
                    'newItems'     => $newItems,
                    'saleItems'    => $saleItems,
                    'historyItems' => $historyItems,
                    'productCount' => $productCount,
                    'brandCount'   => $brandCount,
                    'brands'       => $brands,
                    'articles'     => $articles,
                ]
            );
        }
        
        /**
         * Displays contact page.
         *
         * @return mixed
         */
        public function actionContact()
        {
            $contact = new Feedback();
            return $this->render(
                'contact',
                [
                    'contact' => $contact,
                ]
            );
        }
        
        /**
         * Displays about page.
         *
         * @return mixed
         */
        public function actionAbout()
        {
            return $this->render('about');
        }
        
        /**
         * Action to view robots.txt file dinamycli
         *
         * @return string
         */
        public function actionRobots()
        {
            $response = \Yii::$app->response;
            /**
             * @var Settings $settings
             */
            $settings = Settings::find()
                                ->one();
            $temp = tmpfile();
            fwrite($temp, $settings->robots);
            $meta = stream_get_meta_data($temp);
            $response->format = $response::FORMAT_RAW;
            $response->headers->set('Content-Type', 'text/plain');
            return $this->renderFile($meta[ 'uri' ]);
        }
        
        public function actionFeedback()
        {
            Yii::$app->response->format = Response::FORMAT_JSON;
            if (empty(Yii::$app->request->post())) {
                throw new BadRequestHttpException();
            } else {
                $model = new Feedback(
                    [
                        'scenario' => Feedback::SCENARIO_CALLBACK,
                    ]
                );
                if ($model->load(Yii::$app->request->post()) && $model->save()) {
                    return [
                        'success' => true,
                        'message' => \Yii::t('app', 'Заявка успешно отправлена'),
                        'alert'   => '<div class="alert alert-success">
            <h3>' . \Yii::t('app', 'Заявка успешно отправлена') . '</h3>
            <p>
              ' . \Yii::t(
                                'app',
                                'Спасибо, Ваша заявка успешно отправлена. Наши менеджеры свяжутся с Вами в ближайшее время'
                            ) . 'Success text
            </p>
          </div>',
                    ];
                } else {
                    return [
                        'success' => false,
                        'error'   => $model->errors,
                    ];
                }
            }
        }
    
        public function actionCallback()
        {
            Yii::$app->response->format = Response::FORMAT_JSON;
            $model = new Feedback(
                [
                    'scenario' => Feedback::SCENARIO_CALLBACK,
                ]
            );
            if ($model->load(Yii::$app->request->post()) && $model->save()) {
                \Yii::$app->session->setFlash(
                    'success',
                    Html::tag(
                        'div',
                        Html::tag(
                            'h3',
                            \Yii::t('app', 'Обратный звонок')
                        ) . Html::tag(
                            'p',
                            \Yii::t(
                                'app',
                                'Ваша заявка успешно отправлена. Мы свяжемся с Вами в ближайшее время.'
                            )
                        )
                    )
                );
                return $this->redirect($model->returnUrl ? : [ 'index' ]);
            } else {
                return [
                    'success' => false,
                    'error'   => $model->errors,
                ];
            }
        }
        
        public function actionLogin()
        {
            $serviceName = Yii::$app->getRequest()
                                    ->getQueryParam('service');
            if (isset($serviceName)) {
                /** @var $eauth \nodge\eauth\ServiceBase */
                $eauth = Yii::$app->get('eauth')
                                  ->getIdentity($serviceName);
                $eauth->setRedirectUrl(
                    Yii::$app->getUser()
                             ->getReturnUrl()
                );
                $eauth->setCancelUrl(
                    Yii::$app->getUrlManager()
                             ->createAbsoluteUrl('site/login')
                );
    
                try {
                    if ($eauth->authenticate()) {
                        //                  var_dump($eauth->getIsAuthenticated(), $eauth->getAttributes()); exit;
                        $identity = Customer::findByEAuth($eauth);
                        Yii::$app->getUser()
                                 ->login($identity);
    
                        // special redirect with closing popup window
                        $eauth->redirect();
                    } else {
                        // close popup window and redirect to cancelUrl
                        $eauth->cancel();
                    }
                } catch (ErrorException $e) {
                    // save error to show it later
                    Yii::$app->getSession()
                             ->setFlash('error', 'EAuthException: ' . $e->getMessage());
        
                    // close popup window and redirect to cancelUrl
                    //              $eauth->cancel();
                    $eauth->redirect($eauth->getCancelUrl());
                }
            }
            if (!\Yii::$app->user->isGuest) {
                return $this->redirect([ 'index' ]);
            }
    
            $loginForm = new LoginForm();
            $signupForm = new SignupForm();
            if ($loginForm->load(\Yii::$app->request->post()) && $loginForm->login()) {
                if (!empty($loginForm->returnUrl)) {
                    return $this->redirect($loginForm->returnUrl);
                } else {
                    return $this->redirect([ 'index' ]);
                }
            }
            if ($signupForm->load(\Yii::$app->request->post())) {
                if ($user = $signupForm->signup()) {
                    if (\Yii::$app->getUser()
                                  ->login($user)
                    ) {
                        return $this->redirect([ 'index' ]);
                    }
                }
            }
            return $this->render(
                'login',
                [
                    'loginForm'  => $loginForm,
                    'signupForm' => $signupForm,
                ]
            );
        }
    
        public function actionLogout()
        {
            \Yii::$app->user->logout();
    
            return $this->redirect([ 'index' ]);
        }
    
        /**
         * Requests password reset.
         *
         * @return mixed
         */
        public function actionRequestPasswordReset()
        {
            $model = new PasswordResetRequestForm();
            if ($model->load(Yii::$app->request->post()) && $model->validate()) {
                if ($model->sendEmail()) {
                    Yii::$app->session->setFlash('success', 'Check your email for further instructions.');
    
                    return $this->redirect([ 'index' ]);
                } else {
                    Yii::$app->session->setFlash(
                        'error',
                        'Sorry, we are unable to reset password for the provided email address.'
                    );
                }
            }
    
            return $this->render(
                'requestPasswordResetToken',
                [
                    'model' => $model,
                ]
            );
        }
    
        /**
         * Resets password.
         *
         * @param string $token
         *
         * @return mixed
         * @throws BadRequestHttpException
         */
        public function actionResetPassword($token)
        {
            try {
                $model = new ResetPasswordForm($token);
            } catch (InvalidParamException $e) {
                throw new BadRequestHttpException($e->getMessage());
            }
    
            if ($model->load(Yii::$app->request->post()) && $model->validate() && $model->resetPassword()) {
                Yii::$app->session->setFlash('success', 'New password saved.');
        
                return $this->redirect([ 'index' ]);
            }
    
            return $this->render(
                'resetPassword',
                [
                    'model' => $model,
                ]
            );
        }
    
        public function actionShareBasket()
        {
            $response = \Yii::$app->response;
            $response->format = $response::FORMAT_JSON;
            $model = new ShareBasket();
            if ($model->load(\Yii::$app->request->post()) && $model->validate()) {
                $result = $model->send();
                return $result;
            } else {
                return [
                    'success' => false,
                    'msg'     => \Yii::t('app', 'Email is not set.'),
                ];
            }
        }
    
        public function actionGetBasket($items)
        {
            /**
             * @var \artbox\order\models\Basket $basket
             */
            $basket = \Yii::$app->get('basket');
            $itemArray = Json::decode($items);
            foreach ($itemArray as $variantId => $count) {
                $basket->set($variantId, $count);
            }
            return $this->redirect('/checkout/index');
        }
    }
