<?php
    
    namespace frontend\controllers;
    
    use artbox\catalog\helpers\FilterHelper;
    use artbox\catalog\models\Category;
    use artbox\core\components\SeoComponent;
    use artbox\core\models\Alias;
    use yii\data\ActiveDataProvider;
    use yii\db\ActiveQuery;
    use yii\web\Controller;
    use yii\web\NotFoundHttpException;
    use Yii;
    
    /**
     * Class CategoryController
     *
     * @package frontend\controllers
     */
    class CategoryController extends Controller
    {
        /**
         * Show category by ID
         *
         * @param string $category
         * @param string $filter
         *
         * @return string
         */
        public function actionView($category, $filter = '')
        {
            $model = $this->findModel($category);
            /**
             * @var SeoComponent $seo
             * @var FilterHelper $filterHelper
             */
            $seo = Yii::$app->get('seo');
            $seo->setModel($model->lang);
            $filterHelper = \Yii::$app->get('filter');
            $filterHelper->setFilter($filter);
            $query = $filterHelper->buildQuery()
                                  ->innerJoinWith('category', false)
                                  ->andWhere([ 'product_to_category.category_id' => $model->id ])
                                  ->innerJoinWith('lang')
                                  ->innerJoinWith('variant')
                                  ->with('image', 'variants.image');
            $dataProvider = new ActiveDataProvider(
                [
                    'query'      => $query,
                    'pagination' => [
                        'pageSize' => 18,
                    ],
                    'sort'       => [
                        'attributes'   => [
                            'relevant'   => [
                                'asc'   => [
                                    'product.id' => SORT_DESC,
                                ],
                                'desc'  => [
                                    'product.id' => SORT_DESC,
                                ],
                                'label' => \Yii::t('app', 'по умолчанию'),
                            ],
                            'title_asc'  => [
                                'asc'   => [
                                    'product_lang.title' => SORT_ASC,
                                ],
                                'desc'  => [
                                    'product_lang.title' => SORT_ASC,
                                ],
                                'label' => \Yii::t('app', 'по имени от А до Я'),
                            ],
                            'title_desc' => [
                                'asc'   => [
                                    'product_lang.title' => SORT_DESC,
                                ],
                                'desc'  => [
                                    'product_lang.title' => SORT_DESC,
                                ],
                                'label' => \Yii::t('app', 'по имени от Я до А'),
                            ],
                            'price_asc'  => [
                                'asc'   => [
                                    'variant.price' => SORT_ASC,
                                ],
                                'desc'  => [
                                    'variant.price' => SORT_ASC,
                                ],
                                'label' => \Yii::t('app', 'по цене по возрастанию'),
                            ],
                            'price_desc' => [
                                'asc'   => [
                                    'variant.price' => SORT_DESC,
                                ],
                                'desc'  => [
                                    'variant.price' => SORT_DESC,
                                ],
                                'label' => \Yii::t('app', 'по цене по убыванию'),
                            ],
                        ],
                        'defaultOrder' => [
                            'relevant' => SORT_DESC,
                        ],
                    ],
                ]
            );
            $queryClone = clone $query;
            $queryClone->orderBy = [];
            $queryClone->where = [];
            $queryClone->limit = false;
            $priceMin = $queryClone->select('min(price)')
                                   ->scalar();
            $priceMax = $queryClone->select('max(price)')
                                   ->scalar();
            return $this->render(
                'view',
                [
                    'model'        => $model,
                    'dataProvider' => $dataProvider,
                    'priceMin'     => $priceMin,
                    'priceMax'     => $priceMax,
                ]
            );
        }
        
        /**
         * Find category by ID
         *
         * @param $category
         *
         * @return \artbox\catalog\models\Category
         * @throws \yii\web\NotFoundHttpException
         */
        protected function findModel($category)
        {
            /**
             * @var SeoComponent $seo
             */
            $seo = Yii::$app->get('seo');
            /**
             * @var FilterHelper $filter
             */
            $filter = \Yii::$app->get('filter');
            /**
             * @var Alias $alias
             */
            $alias = Alias::find()
                          ->where([ 'value' => $category ])
                          ->one();
            if (empty($alias)) {
                throw new NotFoundHttpException('Category not found');
            }
            $id = $filter->getIdFromRoute($alias->route);
            /**
             * @var Category $model
             */
            $model = Category::findWithFilters($id)
                             ->with('lang.alias')
                             ->with('categories.lang.alias')
                             ->with(
                                 [
                                     'parent' => function ($query) {
                                         /**
                                          * @var ActiveQuery $query
                                          */
                                         $query->with('lang.alias', 'categories.lang.alias');
                                     },
                                 ]
                             )
                             ->one();
            $seo->setAlias($model->lang->alias);
            if (!empty($model)) {
                if ($model->lang->alias_id !== $seo->aliasId) {
                    throw new NotFoundHttpException('Wrong language');
                }
                return $model;
            } else {
                throw new NotFoundHttpException('Model not found');
            }
        }
    }