<?php
    
    namespace frontend\controllers;
    
    use artbox\order\models\Customer;
    use artbox\order\models\Order;
    use artbox\order\models\PasswordForm;
    use artbox\order\models\Wishlist;
    use yii\data\ActiveDataProvider;
    use yii\db\ActiveQuery;
    use yii\filters\AccessControl;
    use yii\web\Controller;
    use yii\web\NotFoundHttpException;
    use yii\web\Response;
    
    /**
     * Class AccountController
     *
     * @package frontend\controllers
     */
    class AccountController extends Controller
    {
        /**
         * @inheritdoc
         */
        public function behaviors()
        {
            return [
                'access' => [
                    'class' => AccessControl::className(),
                    'rules' => [
                        [
                            'allow' => true,
                            'roles' => [ '@' ],
                        ],
                    ],
                ],
            ];
        }
    
        /**
         * Account main page with orders
         *
         * @return string
         */
        public function actionIndex()
        {
            /**
             * @var Customer $user
             */
            $user = \Yii::$app->user->identity;
            
            $orders = $user->getOrders()
                           ->with(
                               [
                                   'label.lang',
                                   'orderProducts',
                               ]
                           )
                           ->orderBy(
                               [
                                   'id' => SORT_DESC,
                               ]
                           )
                           ->all();
            
            return $this->render(
                'index',
                [
                    'orders' => $orders,
                ]
            );
        }
    
        /**
         * Account detail order page
         *
         * @param $id
         *
         * @return string
         * @throws \yii\web\NotFoundHttpException
         */
        public function actionOrder($id)
        {
            /**
             * @var Customer $user
             */
            $user = \Yii::$app->user->identity;
            /**
             * @var Order $order
             */
            $order = $user->getOrders()
                          ->with('orderProducts.variant.product.lang')
                          ->where(
                              [
                                  'id' => $id,
                              ]
                          )
                          ->one();
    
            if (empty($order)) {
                throw new NotFoundHttpException(\Yii::t('app', 'Order not found'));
            }
    
            return $this->render(
                'order',
                [
                    'order' => $order,
                ]
            );
        }
    
        /**
         * Acount info page
         *
         * @return string
         */
        public function actionAccount()
        {
            $user = \Yii::$app->user->identity;
            return $this->render(
                'account',
                [
                    'userModel'     => $user,
                    'passwordModel' => new PasswordForm(),
                ]
            );
        }
    
        /**
         * Account wishlist page
         *
         * @return string
         */
        public function actionWishlist()
        {
            /**
             * @var Customer $user
             */
            $user = \Yii::$app->user->identity;
            $query = $user->getWishVariants()
                          ->with(
                              [
                                  'lang',
                                  'product' => function (ActiveQuery $query) {
                                      $query->with(
                                          [
                                              'lang',
                                              'image',
                                          ]
                                      );
                                  },
                              ]
                          );
            $dataProvider = new ActiveDataProvider(
                [
                    'query'      => $query,
                    'pagination' => [
                        'pageSize' => 8,
                    ],
                ]
            );
    
            return $this->render(
                'wishlist',
                [
                    'dataProvider' => $dataProvider,
                ]
            );
        }
    
        /**
         * Account change password action
         *
         * @return string|\yii\web\Response
         */
        public function actionChangePassword()
        {
            /**
             * @var Customer $modeluser
             */
            $model = new PasswordForm();
            $modeluser = \Yii::$app->user->identity;
    
            if ($model->load(\Yii::$app->request->post())) {
                if ($model->validate()) {
                    $modeluser->setPassword($model->newpass);
                    if ($modeluser->save()) {
                        return $this->redirect([ 'index' ]);
                    } else {
                        return $this->render(
                            'account',
                            [
                                'userModel'     => $modeluser,
                                'passwordModel' => $model,
                            ]
                        );
                    }
                } else {
                    return $this->render(
                        'account',
                        [
                            'userModel'     => $modeluser,
                            'passwordModel' => $model,
                        ]
                    );
                }
            }
            return $this->render(
                'account',
                [
                    'userModel'     => $modeluser,
                    'passwordModel' => $model,
                ]
            );
    
        }
    
        /**
         * Account change data action
         *
         * @return string|\yii\web\Response
         */
        public function actionChangeData()
        {
            /**
             * @var Customer $model
             */
            $model = \Yii::$app->user->identity;
    
            if ($model->load(\Yii::$app->request->post())) {
                //                VarDumper::dump($model, 10, 1);die();
                $model->markAttributeDirty('birthday');
                if ($model->save()) {
                    return $this->redirect([ 'index' ]);
                }
            }
            return $this->render(
                'account',
                [
                    'userModel'     => $model,
                    'passwordModel' => new PasswordForm(),
                ]
            );
        }
    
        /**
         * Account delete from wishlist action
         *
         * @return array
         */
        public function actionWishlistDelete()
        {
            \Yii::$app->response->format = Response::FORMAT_JSON;
            if (\Yii::$app->request->isPost) {
                $model = Wishlist::find()
                                 ->where(
                                     [
                                         'user_id' => \Yii::$app->request->post('user'),
                                     ]
                                 )
                                 ->andWhere(
                                     [
                                         'variant_id' => \Yii::$app->request->post('variant'),

                                     ]
                                 )
                                 ->one();
                if (!empty($model) && $model->delete()) {
                    return [
                        'success' => true,
                        'message' => \Yii::t('app', 'Товар удален из избранного'),
                    ];
                }
    
                return [
                    'success' => false,
                    'message' => \Yii::t('app', 'Ошибка'),
                ];
            }
            return [];
        }
    }