using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Rendering;
using Microsoft.EntityFrameworkCore;
using Maps.Entities;

namespace maps_core.Controllers
{
    public class RoadController : Controller
    {
        private readonly PostgresDbContext _context;

        public RoadController(PostgresDbContext context)
        {
            _context = context;    
        }

        // GET: Road
        public async Task<IActionResult> Index()
        {
            var postgresDbContext = _context.Road.Include(r => r.RoadType);
            return View(await postgresDbContext.ToListAsync());
        }

        // GET: Road/Details/5
        public async Task<IActionResult> Details(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var road = await _context.Road.SingleOrDefaultAsync(m => m.RoadId == id);
            if (road == null)
            {
                return NotFound();
            }

            return View(road);
        }

        // GET: Road/Create
        public IActionResult Create()
        {
            ViewData["RoadTypeId"] = new SelectList(_context.RoadType, "RoadTypeId", "RoadTypeId");
            return View();
        }

        // POST: Road/Create
        // To protect from overposting attacks, please enable the specific properties you want to bind to, for 
        // more details see http://go.microsoft.com/fwlink/?LinkId=317598.
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Create([Bind("RoadId,AcceptTransferDoc,AcceptanceDoc,AuthorityAct,EconomicValue,HistoricalBackground,Index,LawDoc,Length,Name,RoadTypeId,Value")] Road road)
        {
            if (ModelState.IsValid)
            {
                _context.Add(road);
                await _context.SaveChangesAsync();
                return RedirectToAction("Index");
            }
            ViewData["RoadTypeId"] = new SelectList(_context.RoadType, "RoadTypeId", "RoadTypeId", road.RoadTypeId);
            return View(road);
        }

        // GET: Road/Edit/5
        public async Task<IActionResult> Edit(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var road = await _context.Road.SingleOrDefaultAsync(m => m.RoadId == id);
            if (road == null)
            {
                return NotFound();
            }
            ViewData["RoadTypeId"] = new SelectList(_context.RoadType, "RoadTypeId", "RoadTypeId", road.RoadTypeId);
            return View(road);
        }

        // POST: Road/Edit/5
        // To protect from overposting attacks, please enable the specific properties you want to bind to, for 
        // more details see http://go.microsoft.com/fwlink/?LinkId=317598.
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Edit(int id, [Bind("RoadId,AcceptTransferDoc,AcceptanceDoc,AuthorityAct,EconomicValue,HistoricalBackground,Index,LawDoc,Length,Name,RoadTypeId,Value")] Road road)
        {
            if (id != road.RoadId)
            {
                return NotFound();
            }

            if (ModelState.IsValid)
            {
                try
                {
                    _context.Update(road);
                    await _context.SaveChangesAsync();
                }
                catch (DbUpdateConcurrencyException)
                {
                    if (!RoadExists(road.RoadId))
                    {
                        return NotFound();
                    }
                    else
                    {
                        throw;
                    }
                }
                return RedirectToAction("Index");
            }
            ViewData["RoadTypeId"] = new SelectList(_context.RoadType, "RoadTypeId", "RoadTypeId", road.RoadTypeId);
            return View(road);
        }

        // GET: Road/Delete/5
        public async Task<IActionResult> Delete(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var road = await _context.Road.SingleOrDefaultAsync(m => m.RoadId == id);
            if (road == null)
            {
                return NotFound();
            }

            return View(road);
        }

        // POST: Road/Delete/5
        [HttpPost, ActionName("Delete")]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> DeleteConfirmed(int id)
        {
            var road = await _context.Road.SingleOrDefaultAsync(m => m.RoadId == id);
            _context.Road.Remove(road);
            await _context.SaveChangesAsync();
            return RedirectToAction("Index");
        }

        private bool RoadExists(int id)
        {
            return _context.Road.Any(e => e.RoadId == id);
        }
    }
}
