using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Mvc.Rendering;
using Microsoft.EntityFrameworkCore;
using Maps.Entities;

namespace maps_core.Controllers
{
    public class CrossSectionController : Controller
    {
        private readonly PostgresDbContext _context;

        public CrossSectionController(PostgresDbContext context)
        {
            _context = context;    
        }

        // GET: CrossSection
        public async Task<IActionResult> Index()
        {
            var postgresDbContext = _context.CrossSection.Include(c => c.Region).Include(c => c.Road).Include(c => c.StateCommon).Include(c => c.SurfaceType);
            return View(await postgresDbContext.ToListAsync());
        }

        // GET: CrossSection/Details/5
        public async Task<IActionResult> Details(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var crossSection = await _context.CrossSection.SingleOrDefaultAsync(m => m.CrossSectionId == id);
            if (crossSection == null)
            {
                return NotFound();
            }

            return View(crossSection);
        }

        // GET: CrossSection/Create
        public IActionResult Create()
        {
            ViewData["RegionId"] = new SelectList(_context.Region, "RegionId", "RegionId");
            ViewData["RoadId"] = new SelectList(_context.Road, "RoadId", "RoadId");
            ViewData["StateCommonId"] = new SelectList(_context.StateCommon, "StateCommonId", "StateCommonId");
            ViewData["SurfaceTypeId"] = new SelectList(_context.SurfaceType, "SurfaceTypeId", "SurfaceTypeId");
            return View();
        }

        // POST: CrossSection/Create
        // To protect from overposting attacks, please enable the specific properties you want to bind to, for 
        // more details see http://go.microsoft.com/fwlink/?LinkId=317598.
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Create([Bind("CrossSectionId,Angle,Direction,DistanceEdge,LengthSection,LengthSurface,LocationLeft,LocationRight,RegionId,RoadId,SafetyAvailability,StateCommonId,SurfaceTypeId,TubeAvailability,Width,YearBuild,YearRepair")] CrossSection crossSection)
        {
            if (ModelState.IsValid)
            {
                _context.Add(crossSection);
                await _context.SaveChangesAsync();
                return RedirectToAction("Index");
            }
            ViewData["RegionId"] = new SelectList(_context.Region, "RegionId", "RegionId", crossSection.RegionId);
            ViewData["RoadId"] = new SelectList(_context.Road, "RoadId", "RoadId", crossSection.RoadId);
            ViewData["StateCommonId"] = new SelectList(_context.StateCommon, "StateCommonId", "StateCommonId", crossSection.StateCommonId);
            ViewData["SurfaceTypeId"] = new SelectList(_context.SurfaceType, "SurfaceTypeId", "SurfaceTypeId", crossSection.SurfaceTypeId);
            return View(crossSection);
        }

        // GET: CrossSection/Edit/5
        public async Task<IActionResult> Edit(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var crossSection = await _context.CrossSection.SingleOrDefaultAsync(m => m.CrossSectionId == id);
            if (crossSection == null)
            {
                return NotFound();
            }
            ViewData["RegionId"] = new SelectList(_context.Region, "RegionId", "RegionId", crossSection.RegionId);
            ViewData["RoadId"] = new SelectList(_context.Road, "RoadId", "RoadId", crossSection.RoadId);
            ViewData["StateCommonId"] = new SelectList(_context.StateCommon, "StateCommonId", "StateCommonId", crossSection.StateCommonId);
            ViewData["SurfaceTypeId"] = new SelectList(_context.SurfaceType, "SurfaceTypeId", "SurfaceTypeId", crossSection.SurfaceTypeId);
            return View(crossSection);
        }

        // POST: CrossSection/Edit/5
        // To protect from overposting attacks, please enable the specific properties you want to bind to, for 
        // more details see http://go.microsoft.com/fwlink/?LinkId=317598.
        [HttpPost]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> Edit(int id, [Bind("CrossSectionId,Angle,Direction,DistanceEdge,LengthSection,LengthSurface,LocationLeft,LocationRight,RegionId,RoadId,SafetyAvailability,StateCommonId,SurfaceTypeId,TubeAvailability,Width,YearBuild,YearRepair")] CrossSection crossSection)
        {
            if (id != crossSection.CrossSectionId)
            {
                return NotFound();
            }

            if (ModelState.IsValid)
            {
                try
                {
                    _context.Update(crossSection);
                    await _context.SaveChangesAsync();
                }
                catch (DbUpdateConcurrencyException)
                {
                    if (!CrossSectionExists(crossSection.CrossSectionId))
                    {
                        return NotFound();
                    }
                    else
                    {
                        throw;
                    }
                }
                return RedirectToAction("Index");
            }
            ViewData["RegionId"] = new SelectList(_context.Region, "RegionId", "RegionId", crossSection.RegionId);
            ViewData["RoadId"] = new SelectList(_context.Road, "RoadId", "RoadId", crossSection.RoadId);
            ViewData["StateCommonId"] = new SelectList(_context.StateCommon, "StateCommonId", "StateCommonId", crossSection.StateCommonId);
            ViewData["SurfaceTypeId"] = new SelectList(_context.SurfaceType, "SurfaceTypeId", "SurfaceTypeId", crossSection.SurfaceTypeId);
            return View(crossSection);
        }

        // GET: CrossSection/Delete/5
        public async Task<IActionResult> Delete(int? id)
        {
            if (id == null)
            {
                return NotFound();
            }

            var crossSection = await _context.CrossSection.SingleOrDefaultAsync(m => m.CrossSectionId == id);
            if (crossSection == null)
            {
                return NotFound();
            }

            return View(crossSection);
        }

        // POST: CrossSection/Delete/5
        [HttpPost, ActionName("Delete")]
        [ValidateAntiForgeryToken]
        public async Task<IActionResult> DeleteConfirmed(int id)
        {
            var crossSection = await _context.CrossSection.SingleOrDefaultAsync(m => m.CrossSectionId == id);
            _context.CrossSection.Remove(crossSection);
            await _context.SaveChangesAsync();
            return RedirectToAction("Index");
        }

        private bool CrossSectionExists(int id)
        {
            return _context.CrossSection.Any(e => e.CrossSectionId == id);
        }
    }
}
